-- CategoryManager can take the pain out of managing 2D physics categories
-- it uses named categories instead of numbers, and simplifies if not eliminates the need to coordinate masks and categories separately

CategoryManager = class()

function CategoryManager:init()
    --this code has had to take into account some odd things 2D physics will and won't do, specifically:
    --categories and mask cannot be set to be empty
    --setting categories or mask to {} will have no effect, and the existing tables will be unchanged
    --category and mask assigments are automatically reordered numerically and redundancies are stripped
    --so setting body.categories = {1, 8, 2, 3, 8, 1} results in body.categories = {1, 2, 3, 8}
    --including any number less than 1 will result only in 1 being put in the table
    --including any number over 16 will result only in 16 being put in the table
    --for example, setting category or mask to {0, 800} will become {1, 16}
    --category and mask tables can only be directly assigned, not manipulated
    --so trying to manipulate a categories table through the following sequence of statements:
    --body.categories = {7}
    --body.categories[#body.categories] = 9
    --table.insert(body.categories, 5)
    --would normally result in body.categories being {7, 9, 5}
    --but actually results in body.categories just being {7}, because that was the only direct assignment
    self.errorAsserting = false
    self.reporting = true
    self.categories = tableWithXEmptyTables(16)
    for i, category in ipairs(self.categories) do
        category.mask = {}
        category.index = i
    end
    
    --there needs to be a default category and a separate default mask to store all bodies
    --category 1 stores all bodies and mask 16 also stores all bodies
    --this is because 2D physics won't allow empty categories/mask tables, so if a body is in only one named category/mask and the user wants to remove them from it, it has to have at least one category/mask it can always retain
    --if the same number category and mask were used to store everything, then everything would always collide with everything else, so they have to use different numbers
    self.categories.omniCategory = self.categories[1]
    self.categories.categoryForOmniMask = self.categories[16]
    self.categories.omniMask = self.categories[16].mask
    self.categories[1].name = "omniCategory"
    self.categories[16].name = "categoryForOmniMask"
end

--below are the core functions: putting a body in a category, taking it out, making it detect or ingore a category, making it detect only one category, and verifying whether or not a body can detect a specific category

--putting a body in a category will automatically create that category if it doesn't already exist (and if all of the 14 available categories aren't taken yet)
function CategoryManager:putIn(categoryName, body)
    if not body or not categoryName then return end
    local error = self:makeCategory(categoryName) --only returns string if error happens
    if not error then
        self:addCategoryNumberTo(body, self.categories[categoryName].index)
        self:syncWith(body)
    end
end

--a category name still exists even if all bodies have been removed from it. Removing categories must be done directly using removeCategory(...), below
function CategoryManager:takeOutOf(categoryName, body)
    if not body or not categoryName then return end
    if self.categories[categoryName] == nil then return end
    self:removeCategoryNumberFrom(body, self.categories[categoryName].index)
    self:syncWith(body)
end

--trying to detect a category that doesn't exist has no effect
function CategoryManager:makeDetect(categoryName, body)
    if not body or not categoryName then return end
    if self.categories[categoryName] == nil then return end
    self:addMaskNumberTo(body, self.categories[categoryName].index)
    self:syncWith(body)
end

--ignoring a category that doesn't exist has no effect
function CategoryManager:makeIgnore(categoryName, body)
    if not body or not categoryName then return end
    if self.categories[categoryName] == nil then return end
    self:removeMaskNumberFrom(body, self.categories[categoryName].index)
    self:syncWith(body)
end

--trying to detect only a category that doesn't exist has no effect, and whatever the body currently detects will stay unchanged
function CategoryManager:makeDetectOnly(categoryName, body)
    if not body or not categoryName then return end
    if self.categories[categoryName] ~= nil then
        body.mask = {self.categories[categoryName].index}
        self:syncWith(body)
    end
end

--returns true or false only if the category exists. Returns nil if either the body or category don't exist.
function CategoryManager:verifyBodyDetects(categoryName, body)
    if not body or not categoryName then return end
    if self.categories[categoryName] == nil then return false end
    return tableHas(self.categories[categoryName].mask, body)
end

--below are convenience functions for managing multiple bodies and categories at once

function CategoryManager:putTheseIn(categoryName, bodies)
    for _, body in ipairs(bodies) do
        self:putIn(categoryName, body)
    end
end

function CategoryManager:makeIgnoreAll(body)
    if not body then return end
    body.mask = {16}
    self:syncWith(body)
end

function CategoryManager:makeDetectAll(body)
    if not body then return end
    body.mask = {1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16}
    self:syncWith(body)
end

function CategoryManager:makeBodiesDetect(categoryNames, bodies)
    if not bodies or not categoryNames then return end
    for _, body in ipairs(bodies) do
        for _, name in ipairs(categoryNames) do
            self:makeDetect(name, body)
        end
    end
end

function CategoryManager:makeBodiesDetectOnly(categoryNames, bodies)
    if not bodies or not categoryNames then return end
    for _, body in ipairs(bodies) do
        self:makeIgnoreAll(body)
    end
    self:makeBodiesDetect(categoryNames, bodies)
end

--below are the functions for creating and removing categories directly
--there can only be 14 named categories, and once all have been used, you will have to remove one before creating any more

--you can create a category on its own before putting any bodies in it
function CategoryManager:makeCategory(name)
    if not name then return end
    if self.categories[name] ~= nil then return end
    local available = self:firstUnusedCategoryTable()
    if available then
        self.categories[name] = available
        available.name = name
    else
        local error = "CategoryManager.makeCategory(...): WARNING: no unused categories, category not made"
        self:report(error, self.errorAsserting)
        return error
    end
end

--removing a category name also removes that category and mask number from all managed bodies
function CategoryManager:removeCategory(categoryName)
    if not categoryName then return end
    local errorString
    if categoryName == "omniCategory" then
        errorString = "CategoryManager.removeCategory(...): WARNING: attempted to remove 'omniCategory' which is reserved for use by CategoryManager"
    elseif categoryName == "omniMask" then
        errorString = "CategoryManager.removeCategory(...): WARNING: attempted to remove 'omniMask' which is reserved for use by CategoryManager"
    elseif categoryName == "categoryForOmniMask" then
        errorString = "CategoryManager.removeCategory(...): WARNING: attempted to remove 'categoryForOmniMask' which is reserved for use by CategoryManager"       
    elseif not self.categories[categoryName] then
        errorString = "CategoryManager.removeCategory(...): WARNING: attempted to remove category by name that doesn't exist"
    end
    if errorString then
        self:report(errorString, self.errorAsserting)
        return errorString
    end
    local categoryTable = self.categories[categoryName]
    while #categoryTable ~= 0 do
        local body = categoryTable[#categoryTable]
        self:removeCategoryNumberFrom(body, categoryTable.index)
        self:syncWith(body)
    end
    while #categoryTable.mask ~= 0 do
        local body = categoryTable.mask[#categoryTable.mask]
        self:removeMaskNumberFrom(body, categoryTable.index)
        self:syncWith(body)
    end
    categoryTable.name = nil
    self.categories[categoryName] = nil
end

function CategoryManager:firstUnusedCategoryTable()
    for i, category in ipairs(self.categories) do
        --is no body in the category?
        if #category == 0
        --is the category unnamed?
        and category.name == nil then 
            return category 
        end
    end
end

--syncWith(...) does most of the heavy lifting under the hood: it makes sure CategoryManager's management system is in line with a body's categories and mask. For the most part, it makes the manager match the body, and doesnt change the body, with one exception: it enforces keeping 1 in the body's categories and 16 in the body's mask  
function CategoryManager:syncWith(body)
    self:report("syncWith(...) starting, body is: ", body)
    if not body then return end
    for i, categoryTable in ipairs(self.categories) do
        --sync this category
        if tableHas(body.categories, i) then
            insertIfNotIn(self.categories[i], body)
        else
            removeIfIn(self.categories[i], body)
        end
        --sync this category's mask
        if tableHas(body.mask, i) then
            insertIfNotIn(self.categories[i].mask, body)
        else
            removeIfIn(self.categories[i].mask, body)
        end
    end
    --make sure body stays in tables for category 1 and mask 16
    insertIfNotIn(self.categories.omniCategory, body)
    insertIfNotIn(self.categories.omniMask, body)
    --make sure body itself has category 1 and mask 16
    self:addCategoryNumberTo(body, 1)
    self:addMaskNumberTo(body, 16)
end

--******* CategoryManager is intended to control categories using names only, that is, via the functions above
--******* The functions below only use the normal numbers-based 2D physics system, and CategoryManager uses them to translate between its named categories and 2D physics's numbered categories
--******* Because they only use the numbers-based system, these functions could theoretically be used on their own to control a body's categories/mask directly and avoid names altogether
--******* Be aware that they have not been designed nor tested to be used on their own, so they may or may not be reliable for that purpose

function CategoryManager:addCategoryNumberTo(body, categoryNumber)
    if not body or not categoryNumber then return end
    local combined = {categoryNumber}
    --add all numbers from bodyOrTable.categories into combined
    for i, number in ipairs(body.categories) do
        table.insert(combined, number) 
    end
    --don't have to check for redundant numbers because 2D physics culls them automatically
    body.categories = combined
end

function CategoryManager:removeCategoryNumberFrom(body, number)
    if not body or not number then return end
    if not body then return end
    --if removing the only number in a body's categories, set categories to {1}, because otherwise the number to remove won't get removed
    if #body.categories == 1 and tableHas(body.categories, number) then
        body.categories = {1}
        if self then --when run from tests, self may be nil
            self:report("CategoryManager.removeFromCategoryNumbersOf(...): WARNING: all numbers were removed from body.categories, which causes body.categories to be automatically set to {1}", self.errorAsserting)
            return
        end  
    end
    local dummyTable = {}
    --copy existing numbers
    for i, number in ipairs(body.categories) do
        table.insert(dummyTable, number)
    end
    --remove number from dummy
    removeIfIn(dummyTable, number)
    --replace categories with dummy
    body.categories = dummyTable
end

function CategoryManager:addMaskNumberTo(body, maskNumber)
    if not body or not maskNumber then return end
    local combined = {maskNumber}
    --add all numbers from bodyOrTable.categories into combined
    for i, number in ipairs(body.mask) do
        table.insert(combined, number) 
    end
    --don't have to check for redundant numbers because physics culls them automatically
    body.mask = combined
end

function CategoryManager:removeMaskNumberFrom(body, number)
    if not body or not number then return end
    --if removing the only number, set mask to {16}, because otherwise the number to remove won't get removed
    if #body.mask == 1 and tableHas(body.mask, number) then
        body.mask = {16}
        return
    end
    local dummyTable = {}
    --copy existing numbers
    for i, number in ipairs(body.mask) do
        table.insert(dummyTable, number)
    end
    --remove number from dummy
    removeIfIn(dummyTable, number)
    --replace categories with dummy
    body.mask = dummyTable
end

--remove body from all tables
function CategoryManager:removeFromManager(body)
    if not body then return end
    for i, subTable in ipairs(self.categories) do 
        removeIfIn(subTable, body)
        removeIfIn(subTable.mask, body)
    end
    removeIfIn(self.categories.omniCategory, body)
    removeIfIn(self.categories.omniMask, body)
end

--a function that can be used to print out errors and optionally to halt the program when an error occurs
function CategoryManager:report(message, isError)
    if self.errorAsserting and isError then
        assert(false, message)
    elseif self.errorReporting then
        print(message)
    end
end

--debugging function to print out category contents
function CategoryManager:stringForCategoriesAndMaskOf(bodyOrTable)
    local returnString = ""   
    applyToValueOrArray(bodyOrTable, function(body)
        returnString = returnString..tostring(body).." categories: {"
        for i, number in ipairs(body.categories) do
            returnString = returnString..number
            if i ~= #body.categories then
                returnString = returnString..", "
            end
        end
        returnString = returnString.."}  mask: {"
        for i, number in ipairs(body.mask) do
            returnString = returnString..number
            if i ~= #body.mask then
                returnString = returnString..", "
            end
        end
        returnString = returnString.."}\n"
        return returnString
    end)
    return returnString
end


function testCategoryManager()
    
    CodeaUnit.detailed = true
    CodeaUnit.skip = false
    
    _:describe("Testing CategoryManager", function()
        
        local reporting = true
        local manager
        local bodiesA = {}
        local bodiesB = {}
        
        _:before(function()
            manager = CategoryManager()
            manager.errorReporting = false
        end)
        
        _:after(function()
            for i,body in ipairs(bodiesA) do
                body:destroy()
            end    
            for i,body in ipairs(bodiesB) do
                body:destroy()
            end  
            for i, bodyTable in ipairs(manager.categories) do
                for i,body in ipairs(bodyTable) do
                    body:destroy()
                end  
                for i,body in ipairs(bodyTable.mask) do
                    body:destroy()
                end  
            end  
            bodiesA = {}
            bodiesB = {}
            manager = nil
        end)
        
        function report(...)
            if not reporting then return end
            local args = {...}
            if #args == 1 and type(args[1]) == "table" then
                print("", table.unpack(args[1])) --without the quotes in front table.unpack only prints first element in table
            else
                print(...) 
            end
        end
        
        function fillCategoryNamesOf(categoryManager)
            for i, _ in ipairs(categoryManager.categories) do
                if i ~= 1 then
                    categoryManager.categories[i].name = "category"..i
                    categoryManager.categories["category"..i] = manager.categories[i]
                end
            end
        end
        
        function xNewBodies(number)
            for i = 1, number do
                table.insert(bodiesA, physics.body(CIRCLE, 70))
            end
            return bodiesA
        end
        
        local stringForCategoriesAndMaskOf = function(bodyOrTable)
            local returnString = ""   
            applyToValueOrArray(bodyOrTable, function(body)
                returnString = returnString..tostring(body).." categories: {"
                for i, number in ipairs(body.categories) do
                    returnString = returnString..number
                    if i ~= #body.categories then
                        returnString = returnString..", "
                    end
                end
                returnString = returnString.."}  mask: {"
                for i, number in ipairs(body.mask) do
                    returnString = returnString..number
                    if i ~= #body.mask then
                        returnString = returnString..", "
                    end
                end
                returnString = returnString.."}\n"
                return returnString
            end)
            return returnString
        end
        
        ----------------------------------TESTS----------------------------------
        _:test("syncWith tests", function()
            local newBody = xNewBodies(1)[1]
            newBody.categories = {12}
            manager:syncWith(newBody)
            _:expect("body has been put in category 12", manager.categories[12]).has(newBody)
            
            newBody.categories = {5}
            manager:syncWith(newBody)
            _:expect("if changed to {5}, body has been put in category 5", manager.categories[5]).has(newBody)
            
            _:expect("if changed to {5}, body is no longer in category 12", tableHas(manager.categories[12], newBody) == false).is(true)
            
            _:expect("body is put in omniCategory table if it's not there already", manager.categories.omniCategory).has(newBody)
            
            _:expect("1 is put into newBody.categories if it's not there", newBody.categories).has(1)
            
            newBody.mask = {8}
            manager:syncWith(newBody)
            _:expect("body gets put in mask 8", manager.categories[8].mask).has(newBody)
            
            _:expect("body is not in other masks, like 7 for instance", tableHas(manager.categories[7].mask, newBody) == false).is(true)
            
            _:expect("16 is put into newBody.mask if it's not there", newBody.mask).has(16)
            _:expect("newBody is put into categories[16].mask if it's not there", manager.categories[16].mask).has(newBody)
        end)
        
        local function funkyBox2DSelfOrganizingTableTestOfAddingTo(addFunction, isCategory, retainedNumber, testKindString)             
            local body = xNewBodies(6)[1]
            local body2 = bodiesA[2]
            local preexistingNumbers, numberToAdd = {5, 7, 12}, 8
            local modifiedTable1, modifiedTable2
            local hasAllOldCategories = true
            if isCategory then 
                body.categories = {retainedNumber}
                body2.categories = preexistingNumbers
            else
                body.mask = {retainedNumber}
                body2.mask = preexistingNumbers
            end
            
            addFunction(nil, body, 5)
            addFunction(nil, body2, 8)
            
            if isCategory then 
                modifiedTable1 = body.categories
                modifiedTable2 = body2.categories
            else
                modifiedTable1 = body.mask
                modifiedTable2 = body2.mask
            end
            
            applyToValueOrArray(preexistingNumbers, function(number)
                if not tableHas(modifiedTable2, number) then
                    hasAllOldCategories = false
                end
                return number
            end)
            
            _:expect(testKindString.." contains right number (5)", modifiedTable1).has(5)
            _:expect(testKindString.." retained starting number ("..retainedNumber..")", modifiedTable1).has(retainedNumber)
            _:expect(testKindString.." has new number when numbers are alreadyin table (8)", modifiedTable2).has(8)
            _:expect(testKindString.." has all old numbers {5, 7, 12}", hasAllOldCategories).is(true)
        end
        
        local function funkyBox2DSelfOrganizingTableTestOfRemoveFrom(removeFunction, isCategory, numberIfEmptied, testKindString)  
            local body1 = xNewBodies(11)[1]
            local modifiedTable
            local preexistingNumbers = {1, 3, 12}
            local hasAllOldCategories = true
            if isCategory then
                body1.categories = preexistingNumbers
            else
                body1.mask = preexistingNumbers
            end
            
            removeFunction(nil, body1, 3)
            
            if isCategory then
                modifiedTable = body1.categories
            else
                modifiedTable = body1.mask
            end
            
            applyToValueOrArray({1, 12}, function(number)
                if not tableHas(modifiedTable, number) then
                    hasAllOldCategories = false
                end
                return number
            end)
            
            _:expect("body does not have removed number in "..testKindString, not tableHas(modifiedTable, 3)).is(true)
            _:expect("body has preexisting numbers in "..testKindString, hasAllOldCategories).is(true)
            
            removeFunction(nil, body1, 1)
            removeFunction(nil, body1, 12)
            
            
            if isCategory then
                modifiedTable = body1.categories
            else
                modifiedTable = body1.mask
            end
            
            _:expect("when all are removed, table is set to "..numberIfEmptied, #modifiedTable == 1 and modifiedTable[1] == numberIfEmptied).is(true)
        end
        
        _:test("addCategoryNumberTo tests", function()    
            funkyBox2DSelfOrganizingTableTestOfAddingTo(manager.addCategoryNumberTo, true, 1, "categories")
        end)
        
        _:test("removeCategoryNumberFrom tests", function()
            funkyBox2DSelfOrganizingTableTestOfRemoveFrom(manager.removeCategoryNumberFrom, true, 1, "categories")  
        end)
        
        _:test("addMaskNumberTo tests", function()    
            funkyBox2DSelfOrganizingTableTestOfAddingTo(manager.addMaskNumberTo, false, 16, "mask")
        end)
        
        _:test("removeMaskNumberFrom tests", function()    
            funkyBox2DSelfOrganizingTableTestOfRemoveFrom(manager.removeMaskNumberFrom, false, 16, "mask")
        end)
        
        _:test("makeCategory tests", function()
            local alpha, beta = "alpha", "beta"
            manager:makeCategory(alpha)
            _:expect("makeCategory creates name", manager.categories.alpha ~= nil).is(true)
            
            --category 1 is reserved for omniCategory, So the first category created should be at index 2: 
            _:expect("name hashes to table", manager.categories.alpha).is(manager.categories[2])
            _:expect("category.name hashes to name", tostring(manager.categories.alpha.name)).is(alpha)
            
            --trying to make a name that already exists shouldn't change the table it hashes to
            manager:makeCategory(alpha)
            _:expect("name stays hashed to original table if makeCategory called again with same name", manager.categories.alpha.index).is(2)
            
            fillCategoryNamesOf(manager)
            manager:makeCategory(beta)
            _:expect("does not create name if all categories named", manager.categories.beta == nil).is(true)
        end)  
        
        _:test("removeCategory tests", function()
            local name = "removeCategoryTestName1"
            manager:makeCategory(name)
            local namedTable = manager.categories[name]
            local body1 = xNewBodies(2)[1]
            local body2 = bodiesA[2]
            manager:addCategoryNumberTo(body1, namedTable.index)
            manager:addMaskNumberTo(body2, namedTable.index)
            table.insert(namedTable, body1)
            table.insert(namedTable.mask, body2)
            manager:removeCategory(name)
            _:expect("category name not a key anymore", manager.categories[name] == nil).is(true)
            _:expect("category.name value is nil", namedTable.name == nil).is(true)
            _:expect("namedTable is empty", #namedTable).is(0)
            _:expect("body in category table has number removed", not tableHas(body1.categories, namedTable.index)).is(true)
            _:expect("namedTable.mask is empty", #namedTable.mask).is(0)
            _:expect("body in mask table has number removed", not tableHas(body1.mask, namedTable.index)).is(true)
            _:expect("both bodies still in omniCategory", tableHas(manager.categories.omniCategory, body1) and tableHas(manager.categories.omniCategory, body2)).is(true)
            _:expect("both bodies still in omniMask", tableHas(manager.categories.omniMask, body1) and tableHas(manager.categories.omniMask, body2)).is(true)
            
            local shouldBeErrorString1 = manager:removeCategory("nonexistent name")
            _:expect("trying to remove nonexistent category returns error string", type(shouldBeErrorString1) == "string").is(true)
            
            local shouldBeErrorString2 = manager:removeCategory("omniCategory") 
            _:expect("cannot remove omniCategory", manager.categories.omniCategory ~= nil).is(true)
            _:expect("trying to remove omniCategory returns error string", type(shouldBeErrorString2) == "string").is(true)
            
            local shouldBeErrorString3 = manager:removeCategory("omniMask") 
            _:expect("cannot remove omniMask", manager.categories.omniMask ~= nil).is(true)
            _:expect("trying to remove omniMask returns error string", type(shouldBeErrorString3) == "string").is(true)
            
            local shouldBeErrorString4 = manager:removeCategory("categoryForOmniMask") 
            _:expect("cannot remove categoryForOmniMask", manager.categories.categoryForOmniMask ~= nil).is(true)
            _:expect("trying to remove categoryForOmniMask returns error string", type(shouldBeErrorString4) == "string").is(true)           
        end)
        
        _:test("putIn tests", function()
            local newCat = "newCat"
            xNewBodies(1)
            manager:putIn(newCat, bodiesA[1])
            _:expect("category exists now", manager.categories.newCat ~= nil).is(true)
            _:expect("body is in category table", tableHas(manager.categories.newCat, bodiesA[1])).is(true)
            _:expect("body.category has number of category", tableHas(bodiesA[1].categories, manager.categories.newCat.index)).is(true)
        end)
        
        _:test("takeOutOf tests", function()
            local newCat = "newCat"
            local body = xNewBodies(1)[1]
            manager:putIn(newCat, body)
            local index = manager.categories.newCat.index
            manager:takeOutOf(newCat, body)
            _:expect("body is not in category table", not tableHas(manager.categories.newCat, body)).is(true)
            _:expect("body.category does not have number of category", not tableHas(body.categories, index)).is(true)
        end)
        
        _:test("makeBodyIgnoreAll tests", function()
            local body = xNewBodies(1)[1]
            local hasNone = true
            manager:makeBodyIgnoreAll(body)
            for _, number in ipairs({1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15}) do
                if tableHas(body.mask, number) then
                    hasNone = false
                end
            end
            _:expect("body.mask has no numbers under 16", hasNone).is(true)
            _:expect("body.mask has 16", tableHas(body.mask, 16)).is(true)
        end)
        
        _:test("makeDetect tests", function()
            local body = xNewBodies(1)[1]
            local newCat = "newCat"
            manager:makeCategory(newCat)
            local index = manager.categories.newCat.index
            manager:makeDetect(newCat, body)
            _:expect("body.mask has number of category", tableHas(body.mask, index)).is(true)
            if true then return end
        end)
        
        _:test("applyToValueOrArray tests", function()
            local array1 = {"fo", "ba", "ko", "ja", "loklookloo"}
            applyToValueOrArray(array1, function()
                return "changed"
            end)
            local allChanged = true
            
            for i, v in ipairs(array1) do
                if v ~= "changed" then
                    print(v)
                    allChanged = false
                end
            end
            _:expect("works on string array: ", allChanged).is(true)
            
            local array2 = {{}, {}, {}, {}, {}}
            applyToValueOrArray(array1, function(element1)
                applyToValueOrArray(array2, function(element2)
                    element1 = "changed again"
                    table.insert(element2, element1)
                    return element2
                end)
                return element1
            end)
            allChanged = true
            for i, subArray in ipairs(array2) do
                if not tableHas(subArray, array1[i]) then
                    allChanged = false
                end
                if not subArray[1] == "changed again" then
                    allChanged = false
                end
            end
            _:expect("works on nested arrays: ", allChanged).is(true)
            
            local collectionTable = {}
            applyToValueOrArray(array2, function(element)
                table.insert(collectionTable, element)
            end)
            _:expect("works on references outside inner scope: ", #collectionTable).is(5)
        end)
    end)
end
